const shortid = require("shortid");
const fs = require("fs");

const Link = require("../../models/Link");
const Block = require("../../models/Block.js");
const Socialmedia = require("../../models/Socialmedia");
const Contact = require("../../models/Contact");
const Separator = require("../../models/Separator");
const L_S = require("../../models/L_S");
const L_C = require("../../models/L_C");
const L_Separator = require("../../models/L_Separator.js");
const Banner = require("../../models/Banner");
const Description = require("../../models/Description");
const LinksAddress = require("../../models/LinksAddress");
const Question = require("../../models/Question");

const {
    schemaLink,
    schemaBlock,
    schemaSocialmedia,
    schemaContact,
    schemaSeparator,
    schemaBanner,
    schemaDescription,
    schemaLinkAddress,
    schemaQuestion
} = require("../../validation/user/linkValidation.js");


// link

exports.getLinks = async (req, res, next) => {

    try {
        const links = await Link.findAll({
            where: { userId: req.userId },
            attributes: ['id', 'title', 'description', 'profile', 'thumbnail', 'address']
        });

        if (links.length === 0) {
            const error = new Error("هیچ لینکی یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            res.status(200).json(links);
        }
    } catch (err) {
        next(err);
    }
};

exports.getLink = async (req, res, next) => {
    const address = req.params.address;

    try {
        const link = await Link.findOne({
            where: { address },
            attributes: { exclude: ["createdAt", "updatedAt"] },
        },);


        if (!link) {
            const error = new Error("لینک با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const linkBlocks = await Block.findAll({
                where: { linkId: link.id },
                attributes: { exclude: ["createdAt", "updatedAt"] },
                order: [['sort', 'ASC']]
            },);

            let blocks = JSON.parse(JSON.stringify(linkBlocks));

            for (let i = 0; i < blocks.length; i++) {
                const block = blocks[i];
                let items;
                switch (block.type) {
                    case "socialmedia":
                        items = await L_S.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Socialmedia, as: "data" } })
                        break;
                    case "contact":
                        items = await L_C.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Contact, as: "data" } })
                        break;
                    case "separator":
                        items = await L_Separator.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']], include: { model: Separator, as: "data" } })
                        break;
                    case "banner":
                        items = await Banner.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "description":
                        items = await Description.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "linkAddress":
                        items = await LinksAddress.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    case "question":
                        items = await Question.findAll({ where: { blockId: block.id }, order: [['sort', 'ASC']] })
                        break;
                    default:
                        console.log("test");
                }
                blocks[i] = { ...block, items }
            }


            res.status(200).json({ data: link, blocks });
        }
    } catch (err) {
        next(err);
    }
};

exports.createLink = async (req, res, next) => {
    const profile = req.files ? req.files.profile : {};
    const fileName = `${shortid.generate()}_${profile?.name}`;
    const uploadPath = `./public/uploads/profile/${fileName}`;

    const thumbnail = req.files ? req.files.thumbnail : {};
    const fileName_thumbnail = `${shortid.generate()}_${thumbnail?.name}`;
    const uploadPath_thumbnail = `./public/uploads/thumbnail/${fileName_thumbnail}`;

    try {
        const data = {
            title: req.body.title,
            description: req.body.description,
            address: req.body.address,
            profile: profile,
            thumbnail: thumbnail
        };
        const findLink = await Link.findOne({ where: { address: data.address } });
        if (findLink) {
            const error = new Error(`شناسه (${data.address}) مجاز نیست`);
            error.statusCode = 500;
            throw error;
        }
        if (profile?.name) {
            await schemaLink.validate(data, { abortEarly: false });
            await profile.mv(uploadPath);
            let newLink;
            if (thumbnail?.name) {
                await thumbnail.mv(uploadPath_thumbnail);
                newLink = await Link.create({
                    ...data,
                    userId: +req.userId,
                    profile: fileName,
                    thumbnail: fileName_thumbnail
                });

            } else {
                newLink = await Link.create({
                    ...data,
                    userId: +req.userId,
                    profile: fileName,
                });
            }

            res.status(201).json({ data: newLink, message: "لینک جدید با موفقیت ساخته شد" });

        } else {
            const error = new Error("عکس الزامی می باشد");
            error.statusCode = 500;
            throw error;
        }
        res.status(201).json({ message: "لینک با موفقیت ساخته شد" });
    } catch (err) {
        next(err);
    }
};

exports.editProfile = async (req, res, next) => {
    const profile = req.files ? req.files.profile : {};
    const thumbnail = req.files ? req.files.thumbnail : {};
    const profile_fileName = `${shortid.generate()}_${profile?.name}`;
    const thumbnail_fileName = `${shortid.generate()}_${thumbnail?.name}`;
    const profile_uploadPath = `./public/uploads/profile/${profile_fileName}`;
    const thumbnail_uploadPath = `./public/uploads/thumbnail/${thumbnail_fileName}`;

    const link = await Link.findOne({ where: { address: req.params.address } });

    try {

        if (!link) {
            const error = new Error("لینک با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {


            if (profile?.name && thumbnail?.name)
                await schemaLink.validate({ ...req.body, address: "testttt", profile, thumbnail });
            if (profile?.name && !thumbnail?.name) {
                await schemaLink.validate({ ...req.body, address: "testttt", profile });
            }
            if (!profile?.name && thumbnail?.name) {
                await schemaLink.validate({
                    ...req.body,
                    address: "testttt",
                    profile: {
                        name: "placeholder",
                        size: 0,
                        mimetype: "image/jpeg",
                    },
                    thumbnail,
                });
            } else {
                await schemaLink.validate({
                    ...req.body,
                    address: "testttt",
                    profile: {
                        name: "placeholder",
                        size: 0,
                        mimetype: "image/jpeg",
                    }
                });
            }


            if (profile?.name) {
                fs.unlink(
                    `./public/uploads/profile/${link.profile}`,
                    async (err) => {
                        if (err) {
                            const error = new Error("مشکلی در پاکسازی تصویر پروفایل به وجود آمده است");
                            error.statusCode = 500;
                            throw error;
                        } else {
                            profile.mv(profile_uploadPath);
                        }
                    }
                );
            }
            if (thumbnail?.name) {
                fs.unlink(
                    `./public/uploads/thumbnail/${link.thumbnail}`,
                    async (err) => {
                        if (err) {
                            const error = new Error("مشکلی در پاکسازی تصویر بک گراند به وجود آمده است");
                            error.statusCode = 500;
                            throw error;
                        } else {
                            thumbnail.mv(thumbnail_uploadPath);
                        }
                    }
                );
            }

            const { title, description } = req.body;
            link.title = title;
            link.description = description;
            link.profile = profile?.name ? profile_fileName : link?.profile;
            link.thumbnail = thumbnail?.name ? thumbnail_fileName : link?.thumbnail;
            await link.save();

        }
        res.status(200).json({ data: link, message: "لینک با موفقیت ویرایش شد" });
    } catch (err) {
        next(err);
    }
};

exports.editSettings = async (req, res, next) => {
    const link = await Link.findOne({ where: { address: req.params.address } });

    try {

        if (!link) {
            const error = new Error("لینک با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const { bg, bgBlock, colorBlock, coloBorderBlock, coloText, themeRadius, profileRadius, thumbnailRadius, theme } = req.body;
            link.bg = bg;
            link.bgBlock = bgBlock;
            link.colorBlock = colorBlock;
            link.coloBorderBlock = coloBorderBlock;
            link.coloText = coloText;
            link.theme = theme ? theme : link.theme;
            link.profileRadius = profileRadius ? profileRadius : link.profileRadius;
            link.thumbnailRadius = thumbnailRadius ? thumbnailRadius : link.thumbnailRadius;
            link.themeRadius = themeRadius ? themeRadius : link.themeRadius;
            await link.save();

        }
        res.status(200).json({ data: link, message: "تنظیمات لینک با موفقیت ویرایش شد" });
    } catch (err) {
        next(err);
    }
};

exports.sortLink = async (req, res, next) => {

    const link = await Link.findByPk(req.params.id);

    try {

        if (!link) {
            const error = new Error("لینک با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            for (let i = 0; i < req.body.length; i++) {
                const item = req.body[i];
                await Block.update({ sort: item.sort }, { where: { id: item.id } })
            }

        }
        res.status(200).json({ message: "مرتب سازی با موفقیت انجام شد" });
    } catch (err) {
        next(err);
    }
};

exports.checkAddressLink = async (req, res, next) => {
    try {
        const link = await Link.findOne({ where: { address: req.body.address } });
        if (link) {
            const error = new Error(`مجاز نیست (${req.body.address}) شناسه`);
            error.statusCode = 500;
            throw error;
        } else {
            res.status(200).json({ message: `مجاز است (${req.body.address}) شناسه` });
        }
    } catch (err) {
        next(err);
    }
}

exports.blockAction = async (req, res, next) => {

    const block = req.body;
    const files = req.files;

    try {

        // Check new item information
        await schemaBlock.validate(block, { abortEarly: false });

        // Check the new block


        if (block.id) {
            const oldBlock = await Block.findByPk(block.id);

            // Checking the selected Block
            if (!oldBlock) {
                const error = new Error("بلوکی با شناسه مورد نظر پیدا نشد");
                error.statusCode = 404;
                throw error;
            }

            // Set Items Block            
            await itemsAction(block, block.items, files, next);

            // Edit data Block
            oldBlock.title = block.title;
            block.colorType ? oldBlock.colorType = block.colorType : null
            block.radiusType ? oldBlock.radiusType = block.radiusType : null
            block.widthType ? oldBlock.widthType = block.widthType : null
            block.bg ? oldBlock.bg = block.bg : null
            block.color ? oldBlock.color = block.color : null
            block.coloBorder ? oldBlock.coloBorder = block.coloBorder : null
            block.radius ? oldBlock.radius = block.radius : null
            block.type ? oldBlock.type = block.type : null
            oldBlock.isActive = block.isActive
            oldBlock.save();

            res.status(200).json({ data: oldBlock, message: "بلوک جدید با موفقیت ویرایش شد" });

        } else {

            // Add Block
            const newBlock = await Block.create(block);

            // Set Items Block
            const items = block.type === "banner" ? block.linkAddress : block.items;
            itemsAction(newBlock, items, files, next);


            // Send response
            res.status(201).json({ data: newBlock, message: "بلوک جدید با موفقیت ایجاد شد" });
        }

    } catch (err) {
        next(err);
    }
};

const itemsAction = async (block, items, files, next) => {


    switch (block.type) {
        case "socialmedia":
            await L_S.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                await schemaSocialmedia.validate(items[i], { abortEarly: false });
                await L_S.create({ ...items[i], blockId: block.id, socialmediaId: items[i].data.id });
            }
            break;
        case "contact":
            await L_C.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                await schemaContact.validate(items[i], { abortEarly: false });
                await L_C.create({ ...items[i], blockId: block.id, contactId: items[i].data.id });
            }
            break;
        case "separator":
            await L_Separator.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                await schemaSeparator.validate(items[i], { abortEarly: false });
                await L_Separator.create({ ...items[i], blockId: block.id, separatorId: items[i].data.id });
            }
            break;
        case "banner":

            const banners = Banner.findAll({ where: { blockId: block.id } });
            for (let i = 0; i < banners.length; i++) {
                const banner = banners[i];
                fs.unlink(`./public/uploads/link/banner/${banner.image}`,
                    async (err) => {
                        if (err) {
                            const error = new Error("مشکلی در پاکسازی تصویر پروفایل به وجود آمده است");
                            error.statusCode = 500;
                            throw error;
                        }
                    }
                );
            };
            await Banner.destroy({ where: { blockId: block.id } });

            let images = files.image;
            let allLinkAddress = items;
            if (!Array.isArray(files.image)) {
                images = [files.image]
                allLinkAddress = [allLinkAddress]
            }

            for (let i = 0; i < images.length; i++) {
                const banner = images[i];
                const linkAddress = allLinkAddress[i];
                const fileName = `${shortid.generate()}_${banner?.name}`;
                const uploadPath = `./public/uploads/link/banner/${fileName}`;

                await schemaBanner.validate({ linkAddress: linkAddress ? linkAddress : "", image: banner, sort: i + 1 }, { abortEarly: false });
                await banner.mv(uploadPath);

                await Banner.create({
                    linkAddress: linkAddress,
                    image: fileName,
                    sort: i + 1,
                    blockId: block.id
                })
            }

            break;
        case "description":
            await Description.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                console.log(items[i]);

                await schemaDescription.validate(items[i], { abortEarly: false });
                await Description.create({ ...items[i], blockId: block.id });
            }
            break;
        case "linkAddress":
            await LinksAddress.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                await schemaLinkAddress.validate(items[i], { abortEarly: false });
                await LinksAddress.create({ ...items[i], blockId: block.id });
            }
            break;
        case "question":
            await Question.destroy({ where: { blockId: block.id } })
            for (let i = 0; i < items.length; i++) {
                await schemaQuestion.validate(items[i], { abortEarly: false });
                await Question.create({ ...items[i], blockId: block.id });
            }
            break;
        default:
            console.log("test");
    }

};

const checkHandler = (data, name, next) => {
    if (!data) {
        const error = new Error(`${name} با این شناسه یافت نشد`);
        error.statusCode = 404;
        next(error);
    }
}


exports.deleteBlock = async (req, res, next) => {

    try {
        const block = await Block.findByPk(req.params.id);

        if (!block) {
            const error = new Error("بلوکی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            switch (block.type) {
                case "socialmedia":
                    await L_S.destroy({ where: { blockId: block.id } })
                    break;
                case "contact":
                    await L_C.destroy({ where: { blockId: block.id } })
                    break;
                case "separator":
                    await L_Separator.destroy({ where: { blockId: block.id } })
                    break;
                case "banner":
                    const banners = await Banner.findAll({ where: { blockId: block.id } });
                    for (let i = 0; i < banners.length; i++) {
                        const banner = banners[i];
                        await fs.unlink(`./public/uploads/link/banner/${banner.image}`,
                            async (err) => {
                                if (err) {
                                    const error = new Error("مشکلی در پاکسازی تصویر پروفایل به وجود آمده است");
                                    error.statusCode = 500;
                                    throw error;
                                }
                            }
                        );
                    };
                    await Banner.destroy({ where: { blockId: block.id } })
                    break;
                case "description":
                    await Description.destroy({ where: { blockId: block.id } })
                    break;
                case "linkAddress":
                    await LinksAddress.destroy({ where: { blockId: block.id } })
                    break;
                case "question":
                    await Question.destroy({ where: { blockId: block.id } })
                    break;
                default:
                    console.log("test");
            }


            await Block.destroy({ where: { id: block.id } })
            res.status(200).json({ message: "بلوک شما با موفقیت پاک شد" });
        }
    } catch (err) {
        next(err);
    }
};
