const fs = require("fs");
const shortid = require("shortid");
const QRCode = require('qrcode');

const Menu = require("../../models/Menu.js");
const MenuPlan = require("../../models/MenuPlan.js");
const MenuOrders = require("../../models/MenuOrders.js");
const MenuCategory = require("../../models/MenuCategory.js");
const MenuItme = require("../../models/MenuItem.js");
const Workgroup = require("../../models/Workgroup.js");
const M_MC = require("../../models/M_MC.js");

const { schemaCreateMenu, schemaMenuItem, schemaEditMenu, schemaEditMenuTheme, schemaEditMenuInfo, schemaEditMenuContact, schemaEditMenuCategory, schemaEditMenuLocation, schemaMenuNotification } = require("../../models/secure/menuValidation.js");
const City = require("../../models/City.js");
const Province = require("../../models/Province.js");
const MenuNotification = require("../../models/MenuNotification.js");


// Menu
exports.getMenu = async (req, res, next) => {
    try {
        const menu = await Menu.findOne({
            where: { address: req.params.address, active: true },
            attributes: { exclude: ["active", "workgroupId", "createdAt", "updatedAt"] },
            include: [{
                model: Workgroup,
                as: "workgroup",
                attributes: { exclude: ["createdAt", "updatedAt"] },
            },
            {
                model: M_MC,
                as: "categories",
                attributes: { exclude: ["createdAt", "updatedAt"], },
                include: [
                    {
                        model: MenuCategory,
                        as: "category",
                        attributes: { exclude: ["createdAt", "updatedAt"], },
                    },
                    {
                        model: MenuItme,
                        as: "items",
                        attributes: { exclude: ["createdAt", "updatedAt"] },
                    },
                ]
            },
            {
                model: City,
                as: "city",
                include: {
                    model: Province,
                    as: "province",
                }
            },
            {
                model: MenuNotification,
                as: "notification",
            },
            ],
            order: [
                ["categories", 'sort', 'ASC'],
                ["categories", "items", 'sort', 'ASC'],
                ["notification", "id", 'DESC'],
            ],
        });

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        const start = new Date();
        const diffTime = Math.abs(start - menu.dataValues.endDate);
        let diffDays = Math.floor(diffTime / (1000 * 60 * 60 * 24));
        if (Date.parse(start) > Date.parse(menu.dataValues.endDate)) {
            menu.active = false;
            menu.save();
            diffDays = 0;
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        res.status(200).json({ ...menu.dataValues, day: diffDays });
    } catch (err) {
        next(err);
    }
};

exports.getMenus = async (req, res, next) => {

    try {
        const items = await Menu.findAll({
            where: {
                userId: req.userId,
                status: ['1', '2', '3']
            },
            attributes: { exclude: ["workgroupId", "userId", "createdAt", "updatedAt"] },
            include: [{
                model: Workgroup,
                as: "workgroup",
                attributes: { exclude: ["createdAt", "updatedAt"] },
            }],
            order: [
                ['id', 'DESC']
            ]
        });

        if (items.length == 0) {
            const error = new Error("هیچ منو ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items: items });
    } catch (err) {
        next(err);
    }
};

exports.getMenuOrders = async (req, res, next) => {

    try {

        const items = await MenuOrders.findAll({
            attributes: { exclude: ["authority", "refId", "menuId", "planId", "updatedAt"] },
            include: [{
                model: Menu,
                as: "menu",
                attributes: ["id", "name", "address"],
                where: { userId: req.userId },
            },
            {
                model: MenuPlan,
                as: "plan",
                attributes: { exclude: ["publish", "createdAt", "updatedAt"] },
            }
            ],
            order: [
                ['id', 'DESC']
            ]
        });

        if (items.length == 0) {
            const error = new Error("هیچ سفارشی ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json({ items });
    } catch (err) {
        next(err);
    }
};

exports.getDraftMenu = async (req, res, next) => {

    try {

        const item = await Menu.findOne({
            where: {
                userId: req.userId,
                status: ['0']
            },
            attributes: { exclude: ["workgroupId", "userId", "createdAt", "updatedAt"] },
            include: [{
                model: Workgroup,
                as: "workgroup",
                attributes: { exclude: ["createdAt", "updatedAt"] },
            }],
        });

        if (!item) {
            const error = new Error("هیچ منو پیش نویس ثبت نشده است");
            error.statusCode = 404;
            throw error;
        }
        res.status(200).json(item);
    } catch (err) {
        next(err);
    }
};

exports.getQRMenu = async (req, res, next) => {
    try {

        const menu = await Menu.findOne({ where: { address: req.params.address, active: true }, })

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        }

        var opts = {
            errorCorrectionLevel: 'H',
            type: 'image/png',
            quality: 1,
            margin: 0,
            width: 400,
            color: {
                dark: req.body.light ? '#fff' : '#000000ff',
                light: '#0000'
            }
        }

        const url = `https://m-link.ir/menu/${menu.dataValues.address}`;
        await QRCode.toDataURL(url, opts, function (err, image) {
            if (err) throw err
            res.status(200).json({ url, QR: image });
        })
    } catch (err) {
        next(err);
    }
};

exports.buyMenuPlan = async (req, res, next) => {

    const menuId = req.body.menuId;
    try {

        const plan = await MenuPlan.findByPk(req.body.planId);
        if (!plan) {
            const error = new Error(`پلن منو با شناسه ${req.body.planId} وجود ندارد `);
            error.statusCode = 404;
            throw error;
        }

        const discountAmount = (plan.dataValues.price * plan.dataValues.discount) / 100;
        const price = plan.dataValues.price - discountAmount;
        const orderCode = await Math.round(Math.random() * 100000000);


        if (menuId) {
            const menu = await Menu.findOne({ where: { id: menuId, status: ['0', '2'] } });
            if (!menu) {
                const error = new Error(`منو با شناسه ${menuId} وجود ندارد `);
                error.statusCode = 500;
                throw error;
            }

            await MenuOrders.create({
                orderCode,
                price,
                isPay: true,
                planId: req.body.planId,
                menuId,
                userId: req.userId
            });

            if (menu.dataValues.status === '2') {
                let newEnd;

                const planDay = plan.dataValues.day;
                const end = new Date(menu.dataValues.endDate);
                const today = new Date();
                today.setUTCHours(23, 59, 59, 0);

                if (end >= today) {
                    newEnd = new Date(new Date(menu.dataValues.endDate).setDate(end.getDate() + planDay));
                } else {
                    newEnd = new Date(new Date(menu.dataValues.endDate).setDate(today.getDate() + planDay));
                }

                menu.endDate = newEnd;
                menu.active = true;
                menu.save();
            } else {
                menu.status = '1';
                menu.name = req.body.name;
                menu.address = req.body.address;
                menu.save();
            }

        } else {
            await schemaCreateMenu.validate(req.body, { abortEarly: false });
            const address = await Menu.findOne({ where: { address: req.body.address } });
            if (address) {
                const error = new Error(`مجاز نیست (${req.body.address}) آدرس مجازی`);
                error.statusCode = 500;
                throw error;
            }
            const resultMenu = await Menu.create({ name: req.body.name, address: req.body.address, userId: req.userId, status: '1' });
            await MenuOrders.create({
                orderCode,
                price,
                isPay: true,
                planId: req.body.planId,
                menuId: resultMenu.dataValues.id,
                userId: req.userId
            });
        }

        res.status(201).json({ message: "منو جدید با موفقیت خریداری شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenu = async (req, res, next) => {
    const logo = req.files ? req.files.logo : {};
    const image = req.files ? req.files.image : {};
    const logo_fileName = `${shortid.generate()}_${logo?.name}`;
    const image_fileName = `${shortid.generate()}_${image?.name}`;
    const logo_uploadPath = `./public/uploads/menu/logo/${logo_fileName}`;
    const image_uploadPath = `./public/uploads/menu/image/${image_fileName}`;

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaEditMenu.validate({ ...req.body, logo, image });

            if (logo?.name) {
                if (menu.logo) {
                    fs.unlink(`./public/uploads/menu/logo/${menu.logo}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر لوگو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            } else {
                                await logo.mv(logo_uploadPath);
                            }
                        }
                    );
                } else {
                    await logo.mv(logo_uploadPath);
                }
            }
            if (image?.name) {
                if (menu.image) {
                    fs.unlink(
                        `./public/uploads/menu/image/${menu.image}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر منو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            } else {
                                image.mv(image_uploadPath);
                            }
                        }
                    );
                } else {
                    image.mv(image_uploadPath);
                }
            }


            const { name, description, cityId, location, instagram, phoneNumber, workgroupId } = req.body;
            menu.name = name;
            menu.description = description;
            menu.logo = logo?.name ? logo_fileName : menu.logo;
            menu.image = image?.name ? image_fileName : menu.image;
            menu.location = location;
            menu.instagram = instagram;
            menu.phoneNumber = phoneNumber;
            menu.cityId = cityId;
            menu.workgroupId = workgroupId;
            await menu.save();

        }
        res.status(200).json({ message: "منو با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenuTheme = async (req, res, next) => {

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaEditMenuTheme.validate(req.body);

            const { colorTheme, darkMode } = req.body;

            menu.colorTheme = colorTheme;
            menu.darkMode = darkMode;
            await menu.save();

        }
        res.status(200).json({ message: "منو با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};

exports.checkAddressMenu = async (req, res, next) => {
    try {
        const menu = await Menu.findOne({ where: { address: req.body.address } });
        if (menu) {
            const error = new Error(`مجاز نیست (${req.body.address}) شناسه`);
            error.statusCode = 500;
            throw error;
        } else {
            res.status(200).json({ message: `مجاز است (${req.body.address}) شناسه` });
        }
    } catch (err) {
        next(err);
    }
}

exports.editMenuInfo = async (req, res, next) => {
    const logo = req.files ? req.files.logo : {};
    const logo_fileName = `${shortid.generate()}_${logo?.name}`;
    const logo_uploadPath = `./public/uploads/menu/logo/${logo_fileName}`;

    const image = req.files ? req.files.image : {};
    const image_fileName = `${shortid.generate()}_${image?.name}`;
    const image_uploadPath = `./public/uploads/menu/image/${image_fileName}`;

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaEditMenuInfo.validate({ ...req.body, logo, image });

            if (logo?.name) {
                if (menu.logo) {
                    fs.unlink(`./public/uploads/menu/logo/${menu.logo}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر لوگو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            } else {
                                await logo.mv(logo_uploadPath);
                            }
                        }
                    );
                } else {
                    await logo.mv(logo_uploadPath);
                }
            }

            if (image?.name) {
                if (menu.image) {
                    fs.unlink(`./public/uploads/menu/image/${menu.image}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر منو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            } else {
                                image.mv(image_uploadPath);
                            }
                        }
                    );
                } else {
                    image.mv(image_uploadPath);
                }
            }


            if (req.body.removeLogo) {
                if (menu.logo) {
                    fs.unlink(`./public/uploads/menu/logo/${menu.logo}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر لوگو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            }
                        }
                    );
                }
            }

            if (req.body.removeImage) {
                if (menu.image) {
                    fs.unlink(`./public/uploads/menu/image/${menu.image}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر منو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            }
                        }
                    );
                }
            }

            const { name, description, workgroupId, removeLogo, removeImage } = req.body;
            menu.name = name;
            menu.description = description;
            menu.logo = logo?.name ? logo_fileName : removeLogo ? "" : menu.logo;
            menu.image = image?.name ? image_fileName : removeImage ? "" : menu.image;
            menu.workgroupId = workgroupId;
            await menu.save();

        }
        res.status(200).json({ message: "اطلاعات منو با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenuContact = async (req, res, next) => {

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaEditMenuContact.validate(req.body);

            const { instagram, phoneNumber } = req.body;
            menu.instagram = instagram;
            menu.phoneNumber = phoneNumber;
            await menu.save();

        }
        res.status(200).json({ message: "راه های ارتباطی با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenuLocation = async (req, res, next) => {

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaEditMenuLocation.validate(req.body);
            const { location, lat, lng, cityId } = req.body;
            menu.location = location;
            menu.cityId = cityId;
            menu.lat = lat;
            menu.lng = lng;
            await menu.save();

        }
        res.status(200).json({ message: "آدرس با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};


// MenuCategory

exports.addMenuCategory = async (req, res, next) => {
    try {
        const data = {
            menuId: req.body.menuId,
            categoryId: req.body.categoryId,
            sort: req.body.sort
        }
        if (!data.categoryId) {
            const error = new Error("انتخاب دسته بندی منو الزامی می باشد.");
            error.statusCode = 500;
            throw error;
        }
        const item = await M_MC.create(data);
        res.status(201).json({ data: item, message: "دسته بندی با موفقیت اضافه شد" });
    } catch (err) {
        next(err);
    }
};

exports.editMenuCategory = async (req, res, next) => {

    const item = await M_MC.findByPk(req.body.categoryId);

    try {

        if (!item) {
            const error = new Error("انتخاب دسته بندی منو الزامی می باشد.");
            error.statusCode = 500;
            throw error;
        } else {

            await schemaEditMenuCategory.validate(req.body);

            const { description } = req.body;
            item.description = description;
            await item.save();
        }
        res.status(200).json({ message: "دسته بندی با موفقیت ویرایش شد" });

    } catch (err) {
        next(err);
    }
};

exports.deleteMenuCategory = async (req, res, next) => {
    const { categoryId, menuId } = req.params;
    try {
        const category = await M_MC.findByPk(categoryId);

        if (!category) {
            const error = new Error("دسته بندی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const items = await MenuItme.findAll({ where: { categoryId } });
            console.log(items);

            for (let i = 0; i < items.length; i++) {
                const item = items[i];
                if (item.image) {
                    await fs.unlink(`./public/uploads/menu/items/${item.image}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر پروفایل به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            }
                        }
                    );
                }
            };

            await MenuItme.destroy({ where: { categoryId } });
            await M_MC.destroy({ where: { id: categoryId } });
            res.status(200).json({ message: "دسته بندی شما با موفقیت پاک شد" });
        }

    } catch (err) {
        next(err);
    }
};

exports.activeMenuCategory = async (req, res, next) => {

    const item = await M_MC.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("دسته بندی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            item.active = !item.active;
            await item.save();
            res.status(200).json({ message: "وضعیت دسته بندی با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.sortMenuCategory = async (req, res, next) => {

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const myArr = req.body.items;

            myArr.forEach(async (item) => {
                await M_MC.update({ sort: item.sort }, { where: { id: item.id } })
            })

        }
        res.status(200).json({ message: "مرتب سازی با موفقیت انجام شد" });
    } catch (err) {
        next(err);
    }
};


// MenuItem

exports.addMenuItem = async (req, res, next) => {
    const image = req.files ? req.files.image : {};
    const image_fileName = `${shortid.generate()}_${image?.name}`;
    const image_uploadPath = `./public/uploads/menu/items/${image_fileName}`;

    try {

        const data = {
            title: req.body.title,
            description: req.body.description,
            price: req.body.price,
            discount: req.body.discount,
            image: image,
            sort: req.body.sort,
            categoryId: req.body.categoryId,
        };

        await schemaMenuItem.validate(data, { abortEarly: false });

        const manu = await Menu.findByPk(req.body.menuId);
        if (!manu) {
            const error = new Error(`منو با شناسه ${req.body.menuId}وجود ندارد `);
            error.statusCode = 404;
            throw error;
        }

        const category = await M_MC.findByPk(data.categoryId);
        if (!category) {
            const error = new Error(`دسته بندی با شناسه ${data.categoryId}وجود ندارد `);
            error.statusCode = 404;
            throw error;
        }

        if (image.name) {
            await image.mv(image_uploadPath);
        }

        const resultMenuItme = await MenuItme.create({ ...data, image: image.name ? image_fileName : "" });

        res.status(201).json({ data: resultMenuItme.dataValues, message: "آیتم جدید با موفقیت اضافه شد" });


    } catch (err) {
        next(err);
    }
};

exports.editMenuItem = async (req, res, next) => {
    const image = req.files ? req.files.image : {};
    const image_fileName = `${shortid.generate()}_${image?.name}`;
    const image_uploadPath = `./public/uploads/menu/items/${image_fileName}`;

    const menuItem = await MenuItme.findByPk(req.params.id);

    try {

        if (!menuItem) {
            const error = new Error("آیتم منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaMenuItem.validate({ ...req.body, sort: 1, image });

            const category = await M_MC.findByPk(req.body.categoryId);
            if (!category) {
                const error = new Error(`دسته بندی با شناسه ${req.body.categoryId} وجود ندارد `);
                error.statusCode = 404;
                throw error;
            }

            if (image.name) {
                if (menuItem.image) {
                    fs.unlink(
                        `./public/uploads/menu/items/${menuItem.image}`,
                        async (err) => {
                            if (err) console.log(err);
                            else {
                                await image.mv(image_uploadPath);
                            }
                        }
                    );
                } else {
                    await image.mv(image_uploadPath);
                }
            }

            if (req.body.removeImage) {
                if (menuItem.image) {
                    fs.unlink(`./public/uploads/menu/items/${menuItem.image}`,
                        async (err) => {
                            if (err) {
                                const error = new Error("مشکلی در پاکسازی تصویر منو به وجود آمده است");
                                error.statusCode = 500;
                                throw error;
                            }
                        }
                    );
                }
            }


            const { title, description, price, discount, categoryId, removeImage } = req.body;
            menuItem.title = title;
            menuItem.description = description;
            menuItem.price = price;
            menuItem.discount = discount;
            menuItem.image = image?.name ? image_fileName : removeImage ? "" : menuItem.image;
            menuItem.categoryId = categoryId;
            await menuItem.save();
            res.status(200).json({ data: menuItem, message: "آیتم منو با موفقیت ویرایش شد" });
        }

    } catch (err) {
        next(err);
    }
};

exports.deleteMenuItem = async (req, res, next) => {

    const menuItem = await MenuItme.findByPk(req.params.id);

    try {

        if (!menuItem) {
            const error = new Error("آیتم منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const filePath = `./public/uploads/menu/items/${menuItem.image}`;

            if (menuItem.image) {
                fs.unlink(filePath, async (err) => {
                    if (err) {
                        const error = new Error(
                            "خطای در پاکسازی عکس منو مربوطه رخ داده است"
                        );
                        error.statusCode = 400;
                        throw error;
                    } else {
                        await MenuItme.destroy({ where: { id: req.params.id } })
                        res.status(200).json({ message: "آیتم منو شما با موفقیت پاک شد" });
                    }
                });
            } else {
                await MenuItme.destroy({ where: { id: req.params.id } })
                res.status(200).json({ message: "آیتم منو شما با موفقیت پاک شد" });
            }




        }

    } catch (err) {
        next(err);
    }
};

exports.activeMenuItem = async (req, res, next) => {

    const item = await MenuItme.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("آیتمی با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            item.active = !item.active;
            await item.save();
            res.status(200).json({ message: "وضعیت آیتم با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.sortMenuItem = async (req, res, next) => {

    const menu = await Menu.findByPk(req.body.menuId);

    try {

        if (!menu) {
            const error = new Error("منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            const myArr = req.body.items;

            myArr.forEach(async (item) => {
                await MenuItme.update({ sort: item.sort }, { where: { id: item.id } })
            })

        }
        res.status(200).json({ message: "مرتب سازی با موفقیت انجام شد" });
    } catch (err) {
        next(err);
    }
};

exports.copyMenuItem = async (req, res, next) => {
    let image_fileName;
    const image_uploadPath = `./public/uploads/menu/items/`;

    try {

        const menuItem = await MenuItme.findByPk(req.body.id);
        if (!menuItem) {
            const error = new Error(`آیتم با شناسه ${req.body.id}وجود ندارد `);
            error.statusCode = 404;
            throw error;
        }


        if (menuItem.image) {
            image_fileName = `${shortid.generate()}_${menuItem.image}`

            fs.copyFile(`${image_uploadPath}${menuItem.image}`, `${image_uploadPath}${image_fileName}`, (err) => {
                if (err) {
                    console.log("Error Found:", err);
                }
            })

        }

        const lastSort = await MenuItme.max('sort', { where: { categoryId: menuItem.categoryId } });

        const data = {
            title: `${menuItem.title} (کپی)`,
            description: menuItem.description,
            price: menuItem.price,
            discount: menuItem.discount,
            image: image_fileName,
            sort: lastSort + 1,
            categoryId: menuItem.categoryId,
        };

        const copy = await MenuItme.create(data);

        res.status(201).json({ data: copy, message: "کپی با موفقیت انجام شد" });


    } catch (err) {
        next(err);
    }
};

exports.changeMenuItemCategory = async (req, res, next) => {

    const menuItem = await MenuItme.findByPk(req.params.id);

    try {

        if (!menuItem) {
            const error = new Error("آیتم منو با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {


            const category = await M_MC.findByPk(req.body.categoryId);
            if (!category) {
                const error = new Error(`دسته بندی با شناسه ${req.body.categoryId} وجود ندارد `);
                error.statusCode = 404;
                throw error;
            } else {
                const lastSort = await MenuItme.max('sort', { where: { categoryId: req.body.categoryId } });
                menuItem.categoryId = req.body.categoryId;
                menuItem.sort = lastSort ? lastSort + 1 : 1
                await menuItem.save();
            }

            res.status(200).json({ data: menuItem, message: "دسته بندی آیتم منو با موفقیت ویرایش شد" });
        }

    } catch (err) {
        next(err);
    }
};



exports.addMenuNotification = async (req, res, next) => {

    try {

        const data = {
            title: req.body.title,
            text: req.body.text,
            type: req.body.type,
            menuId: req.body.menuId,
        };

        await schemaMenuNotification.validate(data, { abortEarly: false });


        const result = await MenuNotification.create(data);

        res.status(201).json({ data: result.dataValues, message: "اطلاعیه جدید با موفقیت اضافه شد" });

    } catch (err) {
        next(err);
    }
};

exports.editMenuNotification = async (req, res, next) => {

    const item = await MenuNotification.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("اطلاعیه با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            await schemaMenuNotification.validate({ ...req.body });

            const { title, text, type } = req.body;
            item.title = title;
            item.text = text;
            item.type = type;
            await item.save();
            res.status(200).json({ data: item, message: "اطلاعیه با موفقیت ویرایش شد" });
        }

    } catch (err) {
        next(err);
    }
};

exports.activeMenuNotification = async (req, res, next) => {

    const item = await MenuNotification.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("اطلاعیه با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            if (item.active === true) {
                item.active = false;
                item.pin = false;
            } else {
                item.active = true;
            }
            await item.save();
            res.status(200).json({ data: item, message: "اطلاعیه با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.pinMenuNotification = async (req, res, next) => {

    const item = await MenuNotification.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("اطلاعیه با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {

            if (item.pin === false) {
                await MenuNotification.update({ pin: false }, { where: { pin: true, menuId: req.body.menuId } });
                item.active = true;
            }

            item.pin = !item.pin;
            await item.save();
            res.status(200).json({ data: item, message: "اطلاعیه با موفقیت ویرایش شد" });
        }
    } catch (err) {
        next(err);
    }
};

exports.deleteMenuNotification = async (req, res, next) => {

    const item = await MenuNotification.findByPk(req.params.id);

    try {

        if (!item) {
            const error = new Error("اطلاعیه با این شناسه یافت نشد");
            error.statusCode = 404;
            throw error;
        } else {
            await MenuNotification.destroy({ where: { id: req.params.id } })
            res.status(200).json({ message: "اطلاعیه با موفقیت پاک شد" });
        }

    } catch (err) {
        next(err);
    }
};

